#!/bin/env python3
#
import math
#
# ALisP Grammar
#
##########################################
#
#   S Y N T A X
#
##########################################
#
# ----------------------------------------
# Grammar
# ----------------------------------------
#
# Expr -> atom | ( Seq )
# Seq -> nil | Expr Seq
#
# Var   | Firsts | Next
# ------|--------|------
# Expr  | a (    | a ( )
# Seq   | a (    | )
#
# Rule            | Dir
# ----------------|-------
# Expr -> atom    | atom
# Expt -> ( Seq ) | (
# Seq -> nil      | )
# Seq -> Expr Seq | a (
#
# ----------------------------------------
# Utility functions
# ----------------------------------------
#
def tokenize(text):
    """
    Converts a string to a list of tokens.
    """
    text = ''.join(ch for ch in text if ch.isprintable())
    text = text.replace("(", " ( ").replace(")", " ) ")
    tokens = text.split(" ")
    tokens = [tok.strip() for tok in tokens]
    tokens = [tok for tok in tokens if len(tok) > 0]
    return tokens
#
def is_atom(symbol):
    """
    Tests if a symbol is an atom.
    """
    return isinstance(symbol, str) and \
        len(symbol) > 0 and \
        (not ' ' in symbol) and \
        (not '(' in symbol) and \
        (not ')' in symbol)
#

def atom_value(atom):
    """
    Gets the value of an atom. BAD BAD NOT GOOD CODE.
    """
    value = None
    try:
        value = float(atom)
        (d, i) = math.modf(value)
        value = int(i) if d == 0 else value
    except:
        value = atom
    return value
#
# ----------------------------------------
# Context LL(1) for parsing
# ----------------------------------------
#
class ContextLL1:
    """Tracks and supports the state of a LL1 parsing process."""

    def __init__(self, word):
        self.word = word
        self.pos = 0

    def is_complete(self):
        """Checks if the parsing process is complete."""
        return self.pos >= len(self.word)

    def next(self):
        """Returns the next (unread) token."""
        if not self.is_complete():
            return self.word[self.pos]
        else:
            return None

    def consume(self, symbol):
        """'Consumes' symbol, advancing if possible."""
        if not self.is_complete() and self.next() == symbol:
            self.pos += 1
        else:
            self.error(
                f"** ERROR ** Can't CONSUME '{symbol}' while \
                expecting '{self.next()}'.")

    def error(self, message):
        """Error handling."""
        raise Exception(
            f"{message}\nWord: {self.word}\nRead: \{self.word[self.pos:]}\nUnread: {self.word[:self.pos]}")

    def __repr__(self):
        """String representation."""
        return f"\"{' '.join(self.word[:self.pos])}\" | \"{' '.join(self.word[self.pos:])}\""
#
# ----------------------------------------
# LL(1) Parsing
# ----------------------------------------
#
#
# Rule            | Dir
# -----------------|-------
# Expr -> atom    | atom
# Expr -> ( Seq ) | (
#
def proc_Expr(context):
    next_symbol = context.next()
    if next_symbol == "(":
        context.consume("(")
        seq = proc_Seq(context)
        context.consume(")")
        return seq
    elif is_atom(next_symbol):
        context.consume(next_symbol)
        return atom_value(next_symbol)
    else:
        context.error(f"** ERROR ** Can't use Expr with next '{next_symbol}'.")
#
# Rule            | Dir
# -----------------|-------
# Seq -> nil      | )
# Seq -> Expr Seq | a (
#
def proc_Seq(context):
    next_symbol = context.next()
    if is_atom(next_symbol) or next_symbol == "(":
        e = proc_Expr(context)
        s = proc_Seq(context)
        return (e,) + s
    elif next_symbol == ")":
        return tuple()
    else:
        context.error(f"** ERROR ** Can't use Seq with next '{next_symbol}'.")


def parse(text):
    word = tokenize(text)
    if all(is_atom(x) or x in "()" for x in word):
        context = ContextLL1(word)
        expr = proc_Expr(context)
        return expr
    else:
        return None
#
##########################################
#
#   S E M A N T I C S
#
##########################################
#
NIL = tuple()
#
QUOTE = "'"
#
BASE_FUNCTIONS = {
    '+': (2, lambda x, y: x + y),
    '-': (2, lambda x, y: x - y),
    '*': (2, lambda x, y: x * y),
    '/': (2, lambda x, y: x / y),
    #
    '>': (2, lambda x, y: x > y),
    '>=': (2, lambda x, y: x >= y),
    '<': (2, lambda x, y: x < y),
    '<=': (2, lambda x, y: x <= y),
    #
    '==': (2, lambda x, y: x == y),
    '!=': (2, lambda x, y: x != y),
    #
    'not': (1, lambda x: not x),
    'and': (2, lambda x, y: x and y),
    'or': (2, lambda x, y: x or y),
    #
    'sin': (1, lambda x: math.sin(x)),
    'asin': (1, lambda x: math.asin(x)),
    #
    'head': (1, lambda x: x[0]),
    'tail': (1, lambda x: x[1:]),
    'nth': (2, lambda n, x: x[n]),
}
#
# ----------------------------------------
# Alisp evaluation
# ----------------------------------------
#
class AlispContext:
    def __init__(self, outer):
        self.outer = outer
        self.local = dict()

    def defines(self, key):
        return key in self.local.keys() or key in self.outer.keys()

    def set(self, var_name, var_value):
        self.local[var_name] = var_value

    def keys(self):
        return self.local.keys() | self.outer.keys()

    def __getitem__(self, key):
        return self.value(key)

    def value(self, name):
        result = None
        #
        if name in self.local.keys():
            result = self.local[name]
        elif name in self.outer.keys():
            result = self.outer[name]
        elif name == 'True':
            return True
        elif name == 'False':
            return False
        elif is_atomic(name): # 42, -4.2, ola
            result = atom_value(name)
        #
        return result
#
class Lambda:
    def __init__(self, args, expr):
        self.args = args
        self.expr = expr

    def __repr__(self):
        return f"fn ({' '.join(self.args)}) -> ({' '.join(str(x) for x in self.expr)})"
#
# ----------------------------------------
# Utility
# ----------------------------------------
#
def is_number(expr):
    return isinstance(expr, int) or isinstance(expr, float)

def is_symbol(expr):
    return isinstance(expr, str) and len(expr) > 0

def is_text(expr):
    return is_symbol(expr) and expr[0] == QUOTE

def is_variable(expr):
    return is_symbol(expr) and expr[0] != QUOTE

def is_atomic(expr):
    return is_number(expr) or is_symbol(expr)

def is_list(expr):
    return isinstance(expr, tuple) and all(is_list(x) or is_atomic(x) for x in expr)
#
def expr_repr(expr):
    if expr is None:
        return "error"
    elif is_atomic(expr) or isinstance(expr, Lambda):
        return str(expr)
    else:
        return f"({' '.join(expr_repr(x) for x in expr)})"
#
# ----------------------------------------
# Quote/Unquote
# ----------------------------------------
#
def q(text):
    if is_variable(text):
        return f"'{text}"
    elif is_text(text):
        return text
    else:
        return None
#
def uq(text):
    if is_text(text):
        return text[1:]
    elif is_variable(text):
        return text
    else:
        return None
#
# ----------------------------------------
# head, tail, cons
# ----------------------------------------
#
def head(expr):
    if is_list(expr):
        return expr[0]
    else:
        return None
#
def tail(expr):
    if is_list(expr):
        return expr[1:]
    else:
        return None
#
# ----------------------------------------
#
# Evaluation
# ----------------------------------------
#
def eval(expr, context=None, ops=BASE_FUNCTIONS):
    if context is None:
        context = AlispContext({})
    #
    #   Atomic
    #
    context_value = context.value(expr)
    if context_value is not None:
        return context_value
    #
    #   Lists
    #
    elif is_list(expr) and expr != NIL:
        #
        #   -----------------
        #
        #   expr = (op args...)
        #
        #   -----------------
        #
        #   Operations  (+ 3 4)
        #   Write       (write ola)
        #   Functions   (fn (x) (* 2 x))
        #   Assign      (set a 2)
        #   While       (while (> x 0) (set x (- x 1)))
        #   If          (if (> x 0) positivo negativo)
        #   Sequences   (seq (set a 1) (+ a 3))
        #
        op = head(expr)
        args = tail(expr)
        #
        #   OPERATIONS  +, *, -, /, ==, !=
        #
        #   (op args...)
        #
        if op in ops.keys():
            arity, func = ops[op]
            if len(args) >= arity:
                val_args = tuple(eval(x, context, ops) for x in args[:arity])
                return func(*val_args)
            else:
                return None
        #
        #   Write
        #
        #   (write expr)
        #
        #
        elif op == 'write' and len(args) == 1:
            value = eval(args[0], context, ops)
            print(expr_repr(value))
            return value
        #
        #   Read
        #
        #   (read)
        #
        elif op == 'read' and len(args) == 0:
            user_input = input()
            user_prog = parse(user_input)
            value = eval(user_prog, context, ops)
            return value
        #
        #   Function
        #
        #   (fn args expr)
        #
        elif op == 'fn' and len(args) == 2:
            fn_args = args[0]
            fn_expr = args[1]
            return Lambda(fn_args, fn_expr)
        #
        #   ASSIGN
        #
        #   (set var val)
        #
        elif op == 'set' and len(args) == 2:
            var_name = uq(args[0])
            var_value = eval(args[1], context, ops)
            context.set(var_name, var_value)
            return var_value
        #
        #   WHILE
        #
        #   (while guard seq)
        #
        elif op == 'while' and len(args) == 2:
            guard = args[0]
            statements = args[1]
            last = NIL
            while eval(guard, context, ops):
                last = eval(statements, context, ops)
            return last
        #
        #   IF
        #
        #   (if cond seq_true seq_false)
        #
        elif op == 'if' and len(args) == 3:
            cond, seq_true, seq_false = args
            if eval(cond, context, ops):
                return eval(seq_true, context, ops)
            else:
                return eval(seq_false, context, ops)
        #
        #   SEQUENCE
        #
        #   (seq instr *others)
        #
        elif op == 'seq':
            if len(args) == 0:
                return NIL
            else:
                seq = eval(args, context, ops)
                return seq[-1]
        #
        #   LIST
        #
        #   (expr1 *others)
        #
        elif is_list(op):
            val = eval(op, context, ops)
            vals = tuple([eval(xi, context, ops) for xi in args])
            expr2 = (val,) + vals
            # return eval(expr2, context, ops)
            if val == op:
                return expr2
            else:
                return eval(expr2, context, ops)
        #
        #   VAR
        #
        #   (VAR *others)
        #
        elif context.defines(op):
            val = eval(op, context, ops)
            if isinstance(val, Lambda):
                arity = len(val.args)
                if arity <= len(args):
                    local_context = AlispContext(context)
                    for i in range(arity):
                        local_context.set(
                            val.args[i],
                            eval(args[i], context, ops))
                    return eval(val.expr, local_context, ops)
                else:
                    return None
            else:
                expr2 = (val, ) + args
                return eval(expr2, context, ops)
        else:
            return (op,) + eval(args, context, ops)
    else:
        return expr
#
##########################################
#
#   U   S   A   G   E
#
##########################################
#
def repl(start_prog="(Welcome to the ALisP Read-Eval-Print Loop. Type 'quit' to, well, quit.)", start_context={}):
    context = AlispContext(start_context)
    input_text = start_prog
    while input_text != "quit":
        prog = parse(input_text)
        value = eval(prog, context)
        print(expr_repr(value))
        input_text = input("» ")

def load(filename):
    with open(filename, "rt") as source:
        return " ".join(source.readlines())
#
if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser(description="ALisP")
    parser.add_argument("--repl",
        help="Enter the Read-Eval-Print Loop.",
        action="store_true")
    parser.add_argument("-l", "--load", type=str,
        help="Load and evaluate program from file LOAD.")
    parser.add_argument("-e", "--eval", type=str,
        help="Evaluate program from EVAL.")

    args = parser.parse_args()
    prog_src = None
    #
    #   Load?
    #
    if args.load is not None:
        prog_src = load(args.load)
    #
    #   Eval?
    #
    if args.eval is not None:
        if prog_src is None:
            prog_src = args.eval
        else:
            prog_src = f"(seq {prog_src} {args.eval})"
    #
    #
    #
    if prog_src is None:
        prog_src = "ALisP"
    #
    #   REPL?
    #
    if args.repl:
        repl(prog_src)
    else:
        prog = parse(prog_src)
        print(expr_repr(eval(prog)))
